<?php

namespace App\Models;

use Exception;

//incluir la conexion de base de datos
class Client extends Model
{

    private $tableName = 'clients';
    private $relationTable = 'client_zones';

    //metodo insertar regiustro
    public function insertar($name, $last_name, $document_number, $address, $phone, $email, array $zone_ids)
    {
        $this->beginTransaction();

        try {
            $sql = "INSERT INTO $this->tableName (name, last_name, document_number, address, phone, email) 
                VALUES (?, ?, ?, ?, ?, ?)";
            $arrData = array($name, $last_name, $document_number, $address, $phone, $email);
            $client_id = $this->getReturnId($sql, $arrData);

            if (!empty($zone_ids)) {
                $this->insertClientZones($client_id, $zone_ids);
            }

            $this->commit();
            return $client_id;
        } catch (Exception $e) {
            $this->rollBack();
            throw $e;
        }
    }

    // Método para editar cliente con múltiples zonas
    public function editar($id, $name, $last_name, $document_number, $address, $phone, $email, array $zone_ids)
    {
        // Iniciar transacción
        $this->beginTransaction();

        try {
            // 1. Actualizar datos del cliente
            $sql = "UPDATE $this->tableName SET 
                name=?, last_name=?, document_number=?, address=?, phone=?, email=? 
                WHERE id=?";
            $arrData = array($name, $last_name, $document_number, $address, $phone, $email, $id);
            $this->setData($sql, $arrData);

            // 2. Eliminar relaciones existentes
            $this->deleteClientZones($id);

            // 3. Insertar nuevas relaciones
            if (!empty($zone_ids)) {
                $this->insertClientZones($id, $zone_ids);
            }

            $this->commit();
            return true;
        } catch (Exception $e) {
            $this->rollBack();
            throw $e;
        }
    }

    // Método auxiliar para insertar relaciones cliente-zona
    private function insertClientZones($client_id, array $zone_ids)
    {
        $sql = "INSERT INTO $this->relationTable (client_id, zone_id) VALUES ";
        $placeholders = [];
        $values = [];

        foreach ($zone_ids as $zone_id) {
            $placeholders[] = "(?, ?)";
            $values[] = $client_id;
            $values[] = $zone_id;
        }

        $sql .= implode(', ', $placeholders);
        return $this->setData($sql, $values);
    }

    // Método auxiliar para eliminar relaciones cliente-zona
    private function deleteClientZones($client_id)
    {
        $sql = "DELETE FROM $this->relationTable WHERE client_id = ?";
        return $this->setData($sql, array($client_id));
    }

    // Método para obtener las zonas de un cliente
    public function getClientZones($client_id)
    {
        $sql = "SELECT z.* FROM zones z
            JOIN $this->relationTable cz ON z.id = cz.zone_id
            WHERE cz.client_id = ?";
        return $this->getData($sql, array($client_id));
    }

    //funcion para eliminar datos
    public function eliminar($id)
    {
        $sql = "DELETE FROM $this->tableName WHERE id=?";
        $arrData = array($id);
        return $this->setData($sql, $arrData);
    }

    //metodo para mostrar registros
    public function mostrar(string $id)
    {
        // Obtener datos básicos del cliente
        $sql = "SELECT * FROM clients WHERE id = ?";
        $arrData = array($id);
        $client = $this->getData($sql, $arrData);

        if (!empty($client)) {
            $sql_zones = "SELECT z.id, z.name 
                     FROM zones z
                     INNER JOIN client_zones cz ON z.id = cz.zone_id
                     WHERE cz.client_id = '$id'";
            $zones = $this->getDataAll($sql_zones);

            $client['zones'] = $zones;
        }

        return $client;
    }

    public function mostrarPorDocumento(string $document_number)
    {
        $sql = "SELECT * FROM $this->tableName WHERE document_number=?";
        $arrData = array($document_number);
        return  $this->getData($sql, $arrData);
    }

    public function listar($zone_id = '')
    {
        $sql = "SELECT c.*, 
           GROUP_CONCAT(z.name SEPARATOR ', ') AS zones 
           FROM $this->tableName c
           LEFT JOIN client_zones cz ON c.id = cz.client_id
           LEFT JOIN zones z ON cz.zone_id = z.id";

        if ($zone_id !== '') {
            $sql .= " WHERE cz.zone_id='$zone_id'";
        }

        $sql .= " GROUP BY c.id";

        return $this->getDataAll($sql);
    }

    public function listarParaAsistencia()
    {
        $sql = "SELECT c.*, 
           GROUP_CONCAT(z.name SEPARATOR ', ') AS zones 
           FROM $this->tableName c
           LEFT JOIN client_zones cz ON c.id = cz.client_id
           LEFT JOIN zones z ON cz.zone_id = z.id
           GROUP BY c.id";

        return $this->getDataAll($sql);
    }

    public function listarMedidor()
    {
        $sql = "SELECT c.*, m.id AS meter_id FROM $this->tableName c INNER JOIN meters m ON m.client_id=c.id";
        return  $this->getDataAll($sql);
    }

    public function select()
    {
        $sql = "SELECT * FROM $this->tableName";
        return  $this->getDataAll($sql);
    }

    public function buscar($param)
    {
        $search = "%" . $this->sanitizeInput($param) . "%";

        $sql = "SELECT c.id, CONCAT(c.name, ' ', c.last_name) AS client, 
                   c.document_number, m.series, m.id AS meter_id 
            FROM $this->tableName c 
            INNER JOIN meters m ON m.client_id = c.id
            WHERE c.name LIKE '$search' 
               OR c.last_name LIKE '$search' 
               OR c.document_number LIKE '$search' 
               OR m.series LIKE '$search'";

        return $this->getDataAll($sql);
    }

    // Función auxiliar para sanitizar inputs
    private function sanitizeInput($input)
    {
        // Elimina tags HTML y PHP
        $clean = strip_tags($input);
        // Escapa comillas simples y dobles
        $clean = str_replace("'", "", $clean);
        $clean = str_replace('"', '', $clean);
        // Escapa caracteres especiales de SQL
        $clean = addslashes($clean);

        return $clean;
    }
}
